// Copyright 2022, The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

//! Utility to emit the CDDL for messages passed between HAL and TA.

use kmr_common::crypto;
use kmr_wire::*;
use kmr_wire::{keymint::*, secureclock::*, sharedsecret::*};

fn show_schema<T: kmr_wire::AsCborValue>() {
    if let (Some(n), Some(s)) = (<T>::cddl_typename(), <T>::cddl_schema()) {
        println!("{} = {}", n, s);
    }
}

fn main() {
    // CDDL corresponding to types defined by the AIDL spec.

    // newtype wrappers
    show_schema::<DateTime>();
    show_schema::<kmr_wire::KeySizeInBits>();
    show_schema::<kmr_wire::RsaExponent>();

    // enums
    show_schema::<Algorithm>();
    show_schema::<BlockMode>();
    show_schema::<Digest>();
    show_schema::<EcCurve>();
    show_schema::<crypto::CurveType>();
    show_schema::<ErrorCode>();
    show_schema::<HardwareAuthenticatorType>();
    show_schema::<KeyFormat>();
    show_schema::<KeyOrigin>();
    show_schema::<KeyPurpose>();
    show_schema::<PaddingMode>();
    show_schema::<SecurityLevel>();
    show_schema::<Tag>();
    show_schema::<TagType>();

    // structs
    show_schema::<AttestationKey>();
    // BeginResult omitted as it holds a Binder reference
    show_schema::<Certificate>();
    show_schema::<rpc::DeviceInfo>();
    show_schema::<HardwareAuthToken>();
    show_schema::<KeyCharacteristics>();
    show_schema::<KeyCreationResult>();
    show_schema::<KeyMintHardwareInfo>();
    show_schema::<rpc::EekCurve>();
    show_schema::<rpc::MacedPublicKey>();
    show_schema::<rpc::ProtectedData>();
    show_schema::<rpc::HardwareInfo>();
    show_schema::<TimeStampToken>();
    show_schema::<Timestamp>();
    show_schema::<SharedSecretParameters>();

    // Internal exhaustive enum (instead of `KeyParameter` and `KeyParameterValue` from the HAL).
    show_schema::<KeyParam>();

    // CDDL corresponding to types defined in this crate.

    // enums
    show_schema::<KeyMintOperation>();

    // structs

    show_schema::<GetHardwareInfoRequest>();
    show_schema::<GetHardwareInfoResponse>();
    show_schema::<AddRngEntropyRequest>();
    show_schema::<AddRngEntropyResponse>();
    show_schema::<GenerateKeyRequest>();
    show_schema::<GenerateKeyResponse>();
    show_schema::<ImportKeyRequest>();
    show_schema::<ImportKeyResponse>();
    show_schema::<ImportWrappedKeyRequest>();
    show_schema::<ImportWrappedKeyResponse>();
    show_schema::<UpgradeKeyRequest>();
    show_schema::<UpgradeKeyResponse>();
    show_schema::<DeleteKeyRequest>();
    show_schema::<DeleteKeyResponse>();
    show_schema::<DeleteAllKeysRequest>();
    show_schema::<DeleteAllKeysResponse>();
    show_schema::<DestroyAttestationIdsRequest>();
    show_schema::<DestroyAttestationIdsResponse>();
    show_schema::<BeginRequest>();
    show_schema::<InternalBeginResult>(); // Special case
    show_schema::<EarlyBootEndedRequest>();
    show_schema::<EarlyBootEndedResponse>();
    show_schema::<ConvertStorageKeyToEphemeralRequest>();
    show_schema::<ConvertStorageKeyToEphemeralResponse>();
    show_schema::<GetKeyCharacteristicsRequest>();
    show_schema::<GetKeyCharacteristicsResponse>();
    show_schema::<UpdateAadRequest>();
    show_schema::<UpdateAadResponse>();
    show_schema::<UpdateRequest>();
    show_schema::<UpdateResponse>();
    show_schema::<FinishRequest>();
    show_schema::<FinishResponse>();
    show_schema::<AbortRequest>();
    show_schema::<AbortResponse>();

    show_schema::<GetRpcHardwareInfoRequest>();
    show_schema::<GetRpcHardwareInfoResponse>();
    show_schema::<GenerateEcdsaP256KeyPairRequest>();
    show_schema::<GenerateEcdsaP256KeyPairResponse>();
    show_schema::<GenerateCertificateRequestRequest>();
    show_schema::<GenerateCertificateRequestResponse>();
    show_schema::<GenerateCertificateRequestV2Request>();
    show_schema::<GenerateCertificateRequestV2Response>();

    show_schema::<GetSharedSecretParametersRequest>();
    show_schema::<GetSharedSecretParametersResponse>();
    show_schema::<ComputeSharedSecretRequest>();
    show_schema::<ComputeSharedSecretResponse>();

    show_schema::<GenerateTimeStampRequest>();
    show_schema::<GenerateTimeStampResponse>();

    // Autogenerated enums
    show_schema::<PerformOpReq>();
    show_schema::<PerformOpRsp>();

    // Overall response structure
    show_schema::<PerformOpResponse>();
}
